import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/auth";
import path from "path";
import { promises as fs } from "fs";

type ToolItem = {
  name: string;
  slug: string;
  path: string;
  icon?: string;
  cate?: string;
};

const toolsDir = path.join(process.cwd(), "public", "tools");
const indexPath = path.join(toolsDir, "index.json");

async function readIndex(): Promise<ToolItem[]> {
  try {
    const buf = await fs.readFile(indexPath, "utf8");
    const data = JSON.parse(buf);
    return Array.isArray(data) ? data : [];
  } catch {
    return [];
  }
}

async function writeIndex(items: ToolItem[]) {
  await fs.mkdir(toolsDir, { recursive: true });
  await fs.writeFile(indexPath, JSON.stringify(items, null, 2), "utf8");
}

export async function DELETE(_req: NextRequest, { params }: { params: { slug: string } }) {
  const session = await getServerSession(authOptions);
  if (session?.user?.role !== "ADMIN") {
    return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
  }

  const slug = params.slug;
  const items = await readIndex();
  const idx = items.findIndex((t) => t.slug === slug);
  if (idx === -1) return NextResponse.json({ error: "Not found" }, { status: 404 });

  // remove file if exists
  try {
    await fs.unlink(path.join(toolsDir, `${slug}.html`));
  } catch {}

  items.splice(idx, 1);
  await writeIndex(items);
  return NextResponse.json({ success: true });
}

export async function PATCH(req: NextRequest, { params }: { params: { slug: string } }) {
  const session = await getServerSession(authOptions);
  if (session?.user?.role !== "ADMIN") {
    return NextResponse.json({ error: "Unauthorized" }, { status: 403 });
  }

  const oldSlug = params.slug;
  const items = await readIndex();
  const idx = items.findIndex((t) => t.slug === oldSlug);
  if (idx === -1) return NextResponse.json({ error: "Not found" }, { status: 404 });

  const ctype = req.headers.get("content-type") || "";
  let name: string | undefined;
  let cate: string | undefined;
  let newSlug: string | undefined;
  let file: File | null = null;

  if (ctype.includes("multipart/form-data")) {
    const form = await req.formData();
    name = (form.get("name") as string) || undefined;
    cate = (form.get("cate") as string) || undefined;
    newSlug = (form.get("slug") as string) || undefined;
    file = (form.get("file") as File | null) ?? null;
  } else {
    const json = await req.json().catch(() => ({}));
    name = json.name;
    cate = json.cate;
    newSlug = json.slug;
  }

  // validate slug
  if (newSlug) {
    newSlug = newSlug
      .toLowerCase()
      .trim()
      .replace(/[^a-z0-9\-\s]/g, "")
      .replace(/\s+/g, "-")
      .replace(/-+/g, "-")
      .slice(0, 64);
  }

  const item = items[idx];
  const finalSlug = newSlug && newSlug !== oldSlug ? newSlug : oldSlug;

  // If slug changed, ensure unique and rename file
  if (finalSlug !== oldSlug) {
    if (items.some((t) => t.slug === finalSlug)) {
      return NextResponse.json({ error: "Slug already exists" }, { status: 409 });
    }
    try {
      await fs.rename(path.join(toolsDir, `${oldSlug}.html`), path.join(toolsDir, `${finalSlug}.html`));
    } catch {}
    item.slug = finalSlug;
    item.path = `/tools/${finalSlug}.html`;
  }

  if (typeof name === "string" && name.trim()) item.name = name.trim();
  if (typeof cate === "string") item.cate = cate.trim() || undefined;

  // If file uploaded, overwrite html
  if (file) {
    const arrayBuffer = await file.arrayBuffer();
    await fs.writeFile(path.join(toolsDir, `${item.slug}.html`), Buffer.from(arrayBuffer));
  }

  items[idx] = item;
  await writeIndex(items);
  return NextResponse.json(item);
}

